var electron = require('electron');

var app = electron.app;
var ipc = electron.ipcMain;
var BrowserWindow = electron.BrowserWindow;

var fs = require('fs');
var url = require('url');
var path = require('path');
var querystring = require('querystring');

var labCheck = require('./helpers/is-lab-edition');
var singleInstance = require('./helpers/singleinstance');

var packageJSON = require('./package.json');

global.bridge = module;
global.ipc = ipc;

module.exports = function start(config) {

	singleInstance(config.socket, function(check) {
		
		if (check === false) {
			
			// There is another app instance. Focus it.
			
			app.on('ready', function() {
				setImmediate(function(){ app.quit() });
			});

			return;
		}

		// Set up
		
		var win = null;
		var openOnStartup = null;
		var started = false;

		var server = config.server;
		var development = process.argv.includes('development');
		var userDataPath = app.getPath('appData') + path.sep + config.id;

		if (development) {
			global.development = true;
			userDataPath += '-dev';
			server = process.argv[process.argv.indexOf('development')+1];
		}

		app.setPath('userData', userDataPath);

		global.labEdition = config.publicKey && labCheck(config.publicKey);
		var updatesEnabled = !global.labEdition;

		// Initialize the addons

		require('./addons/menu')();
		require('./addons/context-menu')();
		require('./addons/sass')();
		require('./addons/sseserver')();
		require('./addons/httpserver')();
		require('./addons/spawn')();
		require('./addons/screenshot')();
		require('./addons/read-write')();
		require('./addons/capture-requests')();
		updatesEnabled && require('./addons/update')(server);

		// Events

		app.on('ready', function() {

			// Todo: we should restore the maximized state

			var windowSize = [1200, 720];
			var windowSizePath = app.getPath('userData') + path.sep + 'windowsize';

			try {
				windowSize = JSON.parse(fs.readFileSync(windowSizePath, 'utf8'));
			} catch(e){}

			var options = {
				show: false,
				title: config.name,
				width: windowSize[0],
				height: windowSize[1],
				backgroundColor: '#202426',
				center: true,
				darkTheme: true,
				defaultFontFamily: 'sansSerif',
				webPreferences: {
					allowRunningInsecureContent: true,
					nodeIntegration: true,
					enableRemoteModule: true
				}
			};

			win = global.browserWindow = new BrowserWindow(options);

			win.on('closed', function(){
				win = null;
			})

			win.on('close', function(){
				try {
					fs.writeFileSync( windowSizePath, JSON.stringify(win.getSize()));
				} catch(e){}
			});

			if (development) {
				win.openDevTools({ detach: true });
			}
			
			var qs = querystring.stringify({
				server,
				version: packageJSON.version
			});

			win.loadURL(url.pathToFileURL(path.join(__dirname, 'app', 'index.html')) + '?' + qs);

			win.webContents.on('dom-ready', function(){
				win.show();

				started = true;

				if (openOnStartup) {
					win.webContents.send('open-file', openOnStartup);
				}
			});

		});

		app.on('web-contents-created', (event, contents) => {
			contents.on('will-navigate', e => e.preventDefault());
			contents.on('will-redirect', e => e.preventDefault());
		});

		app.on('open-file', function(e, path) {
			if (started) {
				win.webContents.send('open-file', [path]);
			}
			else {
				openOnStartup = [path];
			}
		});

		return function(data) {
			// Will be called on every start of a subsequent bootstrap studio instance
			win.webContents.send('open-file', data);
			app.focus();
		};
	});

	app.on('window-all-closed', function() {
		app.quit();
	});
}
