var electron = {};

(function(){

	var remote = require('electron').remote;
	var dialog = remote.dialog;
	var fs = remote.require('fs');
	var zlib = remote.require('zlib');
	var ipc = require('electron').ipcRenderer;

	var browserWindow = remote.getCurrentWindow();
	var webContents = remote.getCurrentWebContents();

	var chunkSessions = {};
	var currentChunkSessionID = 0;

	electron.os = {
		darwin: 'osx', linux: 'linux', win32: 'windows'
	}[remote.process.platform];

	electron.version = process.versions.electron;

	electron.development = !!remote.getGlobal('development');
	electron.labEdition = !!remote.getGlobal('labEdition');

	electron.separator = remote.require('path').sep;

	electron.homePath = remote.app.getPath('home');
	electron.userDataPath = remote.app.getPath('userData');
	electron.appData = remote.app.getPath('appData');
	electron.tempPath = remote.app.getPath('temp');

	// Update system

	ipc.send('start-updater', readSetting('session', {}).token, readSetting('updateStrategy', 'auto'));
	
	electron.checkForUpdates = () => ipc.send('check-for-updates');
	electron.downloadUpdate = () => ipc.send('download-update');
	
	ipc.on('update-available', (e, version) => app.updateIsAvailable(version));
	ipc.on('update-not-available', e => app.updateIsNotAvailable());
	ipc.on('update-downloaded', e => app.updateHasBeenDownloaded());

	// Opening files

	ipc.on('open-file', function(e, path){
		if (!app) return;
		app.onCliCommand(path);
	});

	// Taking screenshots

	ipc.on('screenshot-ready', function(e, dataURL){
		app.screenshotReady(dataURL);
	});

	// SASS compilation

	ipc.on('sass-result', function(e, msg){
		try {
			var message = JSON.parse(msg);
			app.sassCompilationResult(message);
		}
		catch(e){
			console.error(e);
		}
	});

	electron.compileSASS = function(request){
		ipc.send('sass-compile', JSON.stringify(request) + '\n');
	};

	// Will use this to open files from the command line

	electron.commandLineArgs = remote.process.argv;

	// Listen for network requests

	ipc.on('network-request-completed', function(e, arr){
		if(window.app) app.onNetworkRequest.apply(app, arr);
	});

	// Settings using localStorage

	electron.saveSetting = saveSetting;

	function saveSetting(key, data){
		localStorage[key] = JSON.stringify(data);
	};

	electron.readSetting = readSetting;

	function readSetting(key, def){

		if (localStorage.hasOwnProperty(key)) {
			return JSON.parse(localStorage[key]);
		}

		return def || null;
	};

	// Misc

	electron.setZoomFactor = function(zoomFactor) {
		webContents.zoomFactor = zoomFactor;
	}

	electron.getZoomFactor = function() {
		return Promise.resolve(webContents.zoomFactor);
	}

	electron.focusWindow = function(){
		browserWindow.focus();

		if(electron.os == 'osx'){
			browserWindow.restore();
		}
	}

	electron.restart = function(){
		remote.app.relaunch();
		remote.app.exit(0);
	}

	electron.reloadWindow = function(){
		browserWindow.reload();
	}

	if (electron.development) {

		// Log hanging listeners from last time

		if (localStorage.devHangingListeners) {
			console.warn('Внимание, висящи event listeners!');
			var tmp = JSON.parse(localStorage.devHangingListeners);
			console.table(Object.keys(tmp).map(k => ({'Event Type': k, 'Unbound listeners': tmp[k]})));
			delete localStorage.devHangingListeners;
		}

		// Detect event unbound event listeners when reloading in dev

		var startListeners = {};

		setTimeout(() => {
			app.on('context-opened.profiling', () => {
				
				app.off('.profiling');

				setTimeout(() => {
					// Give it a second to start up
					startListeners = collectListeners();
				}, 1000);
			});
		}, 1000);

		electron.reloadWindow = function(){

			// Close all contexts and check for unbound event listeners.
			// This indicates potential memory leaks.

			for (var ctx of app.openedContexts) {
				app.closeContext(ctx);
			}

			// Give it some time so that all scheduled context-closed
			// event handlers are run.

			setTimeout(() => {

				// Record all event listeners and compare them with the pristine state

				var diff = diffListeners(startListeners, collectListeners());

				if (Object.keys(startListeners).length && Object.keys(diff).length) {
					// There are listeners that haven't been freed. Add them to local 
					// storage and they will be displayed on the next load.
					localStorage.devHangingListeners = JSON.stringify(diff);
				}

				browserWindow.reload();
			}, 50);
		}

		function collectListeners() {
			var map = {};

			for (var key in app.pubsub) {
				map[key] = app.pubsub[key].length;
			}

			return map;
		}

		function diffListeners(map1, map2) {
			
			var result = {};

			// We assume that map2 is a superset of all event groups in map1

			for (var key in map2) {

				if (map2[key] - (map1[key] || 0) < 1) {
					continue;
				}

				result[key] = map2[key] - (map1[key] || 0);
			}

			return result;
		}

		function printListeners(map) {

			for (var key in map) {
				map[key] = app.pubsub[key].length;
			}

			return map;
		}
	}

	electron.setFullScreen = function(flag){
		browserWindow.setFullScreen(flag);
	}

	electron.toggleFullScreen = function(){
		browserWindow.setFullScreen(!browserWindow.isFullScreen());
	}

	electron.toggleDevTools = function(){
		browserWindow.toggleDevTools(); 
	}

	electron.addToRecent = function(path){
		if(!path) return;
		remote.app.addRecentDocument(path);
	};

	electron.quit = function(){
		remote.app.quit();
	}

	electron.setTitle = function(title){
		browserWindow.setTitle(title);
	}

	electron.takeScreenshot = function(){
		ipc.send('take-screenshot');
	}

	electron.spawn = function(command, path){
		ipc.send('spawn', command, path);
	}

	electron.spawnEditor = function(command, path){
		ipc.send('spawn-editor', command, path);			
	}

	electron.gzipSync = function(data) {
		return zlib.gzipSync(data);
	}

	// Render workers. Used for screenshotting dom and extracting element sizes.

	var renderWorkerStack = [];
	var callbackMap = {};
	var callbackID = 0;

	window.addEventListener('message', function(msg){
		
		if (msg.data.type == 'worker-ready') {
			var work = renderWorkerStack.pop();
			if(!work) return;

			callbackMap[work.data.id] = work.callback;
			msg.source.postMessage(work.data, '*');
		}
		else if (msg.data.type == 'worker-result') {
			callbackMap[msg.data.id](msg.data.result);
			delete callbackMap[msg.data.id];
		}
	});

	electron.renderHTMLAndRetrieve = function(data, command, size, callback){
		
		// Supported commands are 'screenshot' and 'dom-size'

		data = {
			id: ++callbackID,
			command,
			...data
		};

		renderWorkerStack.push({data, callback});

		window.open(
			'./render-worker.html',
			"",
			`show=no,scrollbars=no,status=no,width=${size.width},height=${size.height},use-content-size=true`
		);

	}

	// Dialog functions

	electron.showFileOpenDialog = function(props, callback){

		app.freezeUI();

		dialog.showOpenDialog({
			title: props.title || 'Open File',
			defaultPath: props.defaultPath || electron.homePath,
			filters: props.filters || [],
			properties: props.properties || ['openFile']
		}).then(result => {
			app.unfreezeUI();
			callback(result.filePaths);
		});
	};

	electron.showFileSaveDialog = function(props, callback){
		app.freezeUI();

		dialog.showSaveDialog({
			title: props.title || 'Save File',
			defaultPath: props.defaultPath || electron.homePath,
			filters: props.filters || []
		}).then(result => {
			app.unfreezeUI();
			callback(result.filePath);
		});

	};

	electron.pathExists = function(path){
		return fs.existsSync(path);
	}

	electron.getFileSize = function(path){
		return fs.statSync(path).size;
	}

	// External editors

	electron.watchPath = function(path, callback) {
		return fs.watch(path, (eventType, filename) => callback(filename));
	}

	electron.mkdirpSync = function(path, callback) {
		return fs.mkdirSync(path, { recursive: true });
	}

	var asyncCallbacks = {}, asyncID = 0;

	ipc.on('write-status', function(e, id, err) {

		if (id in asyncCallbacks) {

			asyncCallbacks[id](err);
			delete asyncCallbacks[id];
		}

	});

	ipc.on('read-file-async-response', function(e, data, id){

		if (id in asyncCallbacks) {
			asyncCallbacks[id](data);
			delete asyncCallbacks[id];
		}

	});

	ipc.on('read-chunk', function(event, id, data) {
		chunkSessions[id] = chunkSessions[id] || "";
		chunkSessions[id] += data;
	});

	ipc.on('close-readable-chunk-session', function(event, id, err) {

		if (id in asyncCallbacks) {

			asyncCallbacks[id](err);

			delete chunkSessions[id];
			delete asyncCallbacks[id];
		}

	});

	electron.readFile = function(path, process){

		try{

			switch(process){
				case 'base64':
					return Promise.resolve(ipc.sendSync('read-file-base64', path));
				break;
				case 'gzip':
					return Promise.resolve(ipc.sendSync('read-file-gzipped', path));
				break;
				case 'gzip-async':

					// 1. Generate async id
					var id = asyncID++;

					// 2. Add it to a map
					
					ipc.send('read-file-gzipped-async', path, id);
					
					return new Promise(function(resolve, reject) {

						asyncCallbacks[id] = function(err) {
							if (err) {
								reject(err);
								return;
							}

							resolve(chunkSessions[id]);
							return;
						};
					
					});

				break;
				case 'raw-async':
					
					// 1. Generate async id
					var id = asyncID++;

					// 2. Add it to a map

					ipc.send('read-file-raw-async', path, id);

					return new Promise(function(resolve, reject){

						asyncCallbacks[id] = function(data){
							if(!data) reject();
							resolve(data);
						};

					});

				break;
				default:
					return Promise.resolve(ipc.sendSync('read-file-raw', path));
				break;
			}
		}
		catch(e){
			return Promise.reject(e);
		}
	}

	electron.createFolders = function(paths, basePath = '') {
		var sep = electron.separator;

		var createdFolders = new Set();

		for (var path of paths) {

			var folders = path.split(sep);

			// Remove the file
			folders.pop();

			var pp = basePath.length ? basePath + '/' : '';

			for (var folder of folders) {
				pp += folder + sep;

				if (createdFolders.has(pp)) {
					// Skip already created folders
					continue;
				}

				if (fs.existsSync(pp)) {
					continue;
				}

				createdFolders.add(pp);

				if (!electron.mkdirSync(pp)) {
					throw new Error('No rights');
				}
			}

		}
	}

	electron.mkdirSync = function(path){

		try { 
			remote.require('fs').mkdirSync(path)
			return true;
		} catch(e){
			// Returns true if the file exist, false on all other errors
			return /EEXIST/.test(e.message);
		}
	};

	electron.deleteFile = function(path) {
		if (electron.pathExists(path)) {
			return new Promise(function(resolve, reject) {
				fs.unlink(path, function(err) {
					if (err) {
						reject(err);
						return;
					}

					resolve(err);
				});
			});
		}

		return Promise.resolve();
	};

	electron.writeFile = function(path, data, process){

		try{

			var result;

			switch(process){
				case 'raw-from-base64':
					result = ipc.sendSync('write-file-raw-from-base64', path, data);
				break;
				case 'gzip':
					result = ipc.sendSync('write-file-gzipped', path, data);
				break;
				case 'raw-async':
					ipc.send('write-file-raw-async', path, data);
					result = true;
				break;
				default:
					result = ipc.sendSync('write-file-raw', path, data);
				break;
			}

			if(result){
				return Promise.resolve();
			}
			else {
				throw new Error();
			}

		}
		catch(e){
			return Promise.reject(e);
		}

	}

	electron.writeFileChunked = function(path, data) {

		var id = asyncID++;
		var chunkMB = window.chunkMB || 1;
		var chunkSize = 1024 * 1024 * chunkMB;

		var numberOfChunks = Math.ceil(data.length / chunkSize);

		ipc.sendSync('create-writable-chunk-session', path, id);

		writeChunk(0);

		function writeChunk(currentChunk) {

			var result = ipc.sendSync('send-file-chunk', id, data.slice(currentChunk * chunkSize, (currentChunk + 1) * chunkSize));

			if (currentChunk + 1 > numberOfChunks) {

				ipc.sendSync('close-writable-chunk-session', id);

				return;
			}

			setTimeout(function() {
				writeChunk(currentChunk + 1);
			}, window.chunkTimeout || 10);

		}

		return new Promise(function(resolve, reject) {
			asyncCallbacks[id] = function(err) {
				if (err) {
					reject(err);
					return;
				}

				resolve();
				return;
			};
		});

	}

	electron.readDataFile = function(name){

		var p = electron.userDataPath + electron.separator + name;

		if(!electron.pathExists(p)){
			return Promise.resolve();
		}

		return electron.readFile(p, 'raw-async');
	};

	electron.writeDataFile = function(name, data){
		var p = electron.userDataPath + electron.separator + name;
		electron.writeFile(p, data, 'raw-async');
	};

	// SSE and HTTP Binding to ports

	var httpServer, sseServer;

	electron.previewPort = 8000;
	electron.ssePort = 5000;

	electron.listenOnNetwork = function(port, cb){

		var cnt = 0;

		httpServer = remote.getGlobal('createHTTPServer')('preview');
		httpServer.start(port, null, function(address) {
			electron.previewPort = address.port;
			if(++cnt == 2) cb(address);
		});

		sseServer = remote.getGlobal('createSSEServer')('sse');
		sseServer.start(null, null, function(address) {
			electron.ssePort = address.port;
			if(++cnt == 2) cb(address);
		});
	};

	electron.stopListeningOnNetwork = function(cb){
		httpServer.stop();
		sseServer.stop();
		setTimeout(cb, 250);
	};

	// Design preview

	ipc.on('preview-request', function(e, parsed, headers){
		
		app.handlePreviewRequest(parsed.pathname, parsed, headers).then(function(message){

			// Possible values for message:
			// { decode, content, contentType, headers }

			ipc.send('preview-response', parsed.pathname , message);

		}).catch(function(){
			ipc.send('preview-response', parsed.pathname , {
				status: 404,
				content: "Sorry, we can't find this resource."
			});
		});

	});

	// Event stream server

	electron.notifySSEClients = function(message){
		ipc.send('sse-message', message);
	};

	// Generic functions

	electron.getIPAddresses = function(){

		var interfaces = remote.require('os').networkInterfaces();

		var addresses = [];

		for (var k in interfaces) {
			for (var k2 in interfaces[k]) {
				
				var address = interfaces[k][k2];

				if (address.family === 'IPv4') {
					addresses.push(address.address);
				}
			}
		}

		return addresses;
	};

	electron.getHostname = function() {
		return remote.require('os').hostname();
	}

	electron.getDeviceID = function() {

		var interfaces = remote.require('os').networkInterfaces();

		var mac = '00:00:00:00:00:00';
		
		// Common VM Mac prefixes

		var vmMacPrefixes = [ 
			'00:50:56',
			'00:03:FF',
			'00:1C:42',
			'00:0F:4B',
			'00:16:3E',
			'08:00:27',
			'0A:00:27'
		];

		var macs = [];

		for (var k in interfaces) {
			for (var k2 in interfaces[k]) {
				
				var address = interfaces[k][k2];

				if (address.mac == '00:00:00:00:00:00') {
					// Loopback interface
					continue;
				}

				if (vmMacPrefixes.includes(address.mac.toUpperCase().slice(0, 8))) {
					// This is a virtual network interface
					continue;
				}
				
				macs.push(address.mac);
			}
		}

		if (macs.length) {
			macs.sort();
			mac = macs[0];
		}

		return remote.require('crypto')
					.createHash('md5')
					.update(mac)
					.digest("hex");
	}

	electron.openBrowserWindow = function(url){
		require('electron').shell.openExternal(url);
	};

	electron.openFileBrowser = function(path){
		require('electron').shell.openItem(path);
	};

	// Application menu

	ipc.on('menu-click', (e, cbid) => {

		if (!menuCallbackMap.has(cbid)) {
			return;
		}

		menuCallbackMap.get(cbid)();

	});

	var menuCallbackMap = new Map();
	var menuCallbackID = 1;

	electron.setMenu = function(template){

		menuCallbackMap.clear();

		// Replace all callback functions with "pointers"
		walkMenu(template);

		// Send this to electron to display
		ipc.send('set-menu', template);
	};

	function walkMenu(arr) {

		for (var item of arr) {

			for (var key in item) {

				if (Array.isArray(item[key])) {
					walkMenu(item[key]);
					continue;
				}

				if (typeof item[key] == 'function') {
					// Store the callback in the map and replace it with a "pointer"
					menuCallbackMap.set(menuCallbackID, item[key]);
					item[key] = {cbid: menuCallbackID};
					menuCallbackID++;
				}
			}
		}
	}

	electron.setMenu([]); // remove the default menu

	// Clipboard

	var clipboard = require('electron').clipboard;

	electron.clipboardGet = function(){
		return clipboard.readHTML() || clipboard.readText();
	};
		
	electron.clipboardGetText = function(){
		return clipboard.readText();
	};

	electron.clipboardGetHTML = function(){
		return clipboard.readHTML();
	};
	
	electron.clipboardSet = function(text, html){
		return clipboard.write({ text: text, html: html || '' });
	};

	electron.clipboardSetText = function(text){
		return clipboard.writeText(text);
	};

	electron.clipboardSetHTML = function(html){
		return clipboard.writeHTML(html);
	};

})();