var fs = require('fs');
var zlib = require('zlib');

var electron = require('electron');
var app = electron.app;
var ipc = electron.ipcMain;

module.exports = function initialize(){

	ipc.on('read-file-raw', function(e, p){ 
		e.returnValue = fs.readFileSync(p, 'utf8'); 
	});

	ipc.on('read-file-raw-async', function(event, p, id){ 
		try{ 
			var data = fs.readFileSync(p, 'utf8'); 
			event.sender.send('read-file-async-response', data, id); 
		} 
		catch(e){ 
			console.error(e); 
		} 
	});

	ipc.on('read-file-gzipped', function(event, p){ 
		try{ 
			var data = zlib.gunzipSync(fs.readFileSync(p)).toString(); 
			event.returnValue = data; 
		} 
		catch(e){ 
			event.returnValue = false; 
			console.error(e); 
		} 
	});

	ipc.on('read-file-base64', function(e, p){ 
		e.returnValue = fs.readFileSync(p, 'base64'); 
	});

	ipc.on('read-file-gzipped-async', function(event, path, id) {

		try {

			var readable = fs.createReadStream(path);

			readable
				.on('error', (err) => { handleError(event, err, id) })
				.pipe(zlib.createGunzip({chunkSize: 1024*1024}))
				.on('error', (err) => { handleError(event, err, id) })
				.on('data', (chunk) => {
					event.sender.send('read-chunk', id, chunk);
				}).on('end', () => {
					event.sender.send('close-readable-chunk-session', id);
				});

		}
		catch(e){
			console.error(e);
		}

		function handleError(event, err, id) {
			event.sender.send('close-readable-chunk-session', id, err);
		}

	});

		
	// Rename function, taken from graceful-fs

	var rename = fs.rename;

	if (process.platform === "win32") {
		rename = (function (fs$rename) { return function (from, to, cb) {
			var start = Date.now()
			var backoff = 0;
			fs$rename(from, to, function CB (er) {
				if (er
						&& (er.code === "EACCES" || er.code === "EPERM")
						&& Date.now() - start < 60000) {
					setTimeout(function() {
						fs.stat(to, function (stater, st) {
							if (stater && stater.code === "ENOENT")
							fs$rename(from, to, CB);
							else
							cb(er)
						})
					}, backoff)
					if (backoff < 100)
					backoff += 10;
					return;
				}
				if (cb) cb(er)
			})
		}})(rename)
	}

	// Chunk sessions

	var chunkSessions = {};
	var currentChunkSessionID = 1;

	ipc.on('write-file-raw', function(event, path, contents) { 
		try{ 
			fs.writeFileSync(path + '.tmp', contents); 
			rename(path + '.tmp', path, function(e){ if(e) return console.error(e); });
			event.returnValue = true; 
		} 
		catch(e){ 
			event.returnValue = false; 
			console.error(e); 
		} 
	});

	ipc.on('write-file-raw-async', function(event, path, contents) { 
		try{ 
			fs.writeFileSync(path + '.tmp', contents); 
			rename(path + '.tmp', path, function(e){ if(e) return console.error(e); });
		} 
		catch(e){ 
			console.error(e); 
		} 
	});

	ipc.on('write-file-gzipped', function(event, path, contents) { 
		try{ 
			fs.writeFileSync(path + '.tmp', zlib.gzipSync(contents)); 
			rename(path + '.tmp', path, function(e){ if(e) return console.error(e); });
			event.returnValue = true; 
		} 
		catch(e){ 
			event.returnValue = false; 
			console.error(e); 
		} 
	});

	ipc.on('create-writable-chunk-session', function(event, path, id) {
		try {

			chunkSessions[id] = {
				stream: zlib.createGzip(),
				path: path
			}
			
			chunkSessions[id].stream
				.on('error', function(err) { handleError(event, err, id) })
				.pipe(fs.createWriteStream(path + '.tmp'))
				.on('error', function(err) { handleError(event, err, id) })
				.on('finish', function() {
					rename(chunkSessions[id].path + '.tmp', chunkSessions[id].path, function(e) { 

						if(e) {
							console.error(e);
							event.sender.send('write-status', id, e);
						} else {
							event.sender.send('write-status', id);
						}

						delete chunkSessions[id];
					});
				});

			event.returnValue = id;
		}
		catch(e){
			event.returnValue = null;
			console.error(e);
		}

		function handleError(event, err, id) {
			event.sender.send('close-writable-chunk-session', id);
			event.sender.send('write-status', id, err);
		}

	});

	ipc.on('send-file-chunk', function(event, id, contents) {
		try{

			var stream = chunkSessions[id].stream;
			stream.write(contents);
			
			event.returnValue = true;
		}
		catch(e){
			event.returnValue = false;
			console.error(e);
		}
	});

	ipc.on('close-writable-chunk-session', function(event, id) {
		try{
			var stream = chunkSessions[id].stream;
			stream.end();

			event.returnValue = true;
		}
		catch(e){
			delete chunkSessions[id];
			event.returnValue = false;
			console.error(e);
		}
	});

	ipc.on('write-file-raw-from-base64', function(event, path, contents) { 
		try{ 
			fs.writeFileSync(path, Buffer.from(contents, 'base64')); 
			event.returnValue = true; 
		} 
		catch(e){ 
			event.returnValue = false; 
			console.error(e); 
		} 
	});

};